#!/bin/bash

#Copyright (C) 2026 Sanna Favri Glue AG
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.

#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.

#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <https://www.gnu.org/licenses/>.

#HOW IT WORKS:
#Sticky-logic inspired by https://labs.consol.de/de/nagios/check_logfiles
#It will preserve a warning or critical state from the wrapped plugin until it is released with unstick.

VERSION="1.4"
VERDATE="2026-01-26"
SCRIPT_NAME=$(basename "$0")

#For further explanation about exit-codes for Nagios: https://nagios-plugins.org/doc/guidelines.html#PLUGOUTPUT -> Plugin Return Codes
OK=0
WARNING=1
CRITICAL=2
UNKNOWN=3

#Definition of usage function -> is called if options are incorrect or missing in command
function usage() {
  local msg="$*"
  echo "$SCRIPT_NAME v$VERSION ($VERDATE)"
  echo "$SCRIPT_NAME -s <statefile> [wrapper-options] <wrapped_plugin> [wrapped_plugin_options]"
  echo "Options:"
  echo "  -s, --statefile      name of statefile"
  echo "  -t, --tmpdirectory   directory for state file default: /tmp"
  echo "  --sticky             activates status preserving"
  echo "  --unstick            resets preserving"
  if [ -n "$msg" ]; then
    echo "$msg"
  fi
    exit $UNKNOWN
}

#Set default values
tmpdir="/tmp"
sticky=false
unstick=false
state_id=""

#Handling options and arguments
while [[ $# -gt 0 ]]; do
  case $1 in
    -s|--fileforstate) if [ -n "$2" ]; then
        state_id="$2"
      else
        usage "Invalid argument to -s"
      fi
      shift 2
      ;;
    -t|--tmpdirectory) if [ -d "$2" ] && [ -r "$2" ]; then
        tmpdir="$2"
      else
        usage "$2 is not a writable directory"
      fi
      shift 2
      ;;
    --sticky)
        sticky=true;
      shift ;;
    --unstick)
        unstick=true;
      shift ;;
    -h|--help)
      usage
      ;;
    -*)
      usage "Unrecognized option: $1"
      ;;
    *)
      break
      ;;
  esac
done

#validaton
[[ -z "$state_id" ]] && usage Must specify -s

wrapped_command="$@"

#looks for statefile
if [[ -z "$state_id" ]]; then
    echo "UNKNOWN - Missing state identifier (-s)"
    exit $UNKNOWN
fi

statefile="${tmpdir}/.sticky_$(echo "$state_id" | tr '/' '_')"

if [ "$unstick" == true ]; then
  echo "OK" > "$statefile"
  echo "OK -  STATE RESET FOR $state_id"
  exit $OK
fi

if [[ -z "$wrapped_command" ]]; then
    echo "UNKNOWN - No wrapped plugin command provided"
    exit $UNKNOWN
fi

raw_output=$($wrapped_command 2>&1)
exit_code=$?

if [ ! -f "$statefile" ]; then
  echo "OK" > "$statefile"
fi

#applies stickiness and cares about unstick and state preserving
if [ "$sticky" == true ]; then
  if [ $exit_code -eq $CRITICAL ]; then
    echo "CRITICAL" > "$statefile"
  elif [ $exit_code -eq $WARNING ]; then
    saved_now=$(cat "$statefile")
    [ "$saved_now" != "CRITICAL" ] && echo "WARNING" > "$statefile"
  fi

  saved_state=$(cat "$statefile")

  if [ "$saved_state" == "CRITICAL" ]; then
    final_exit_code=$CRITICAL
  elif [ "$saved_state" == "WARNING" ]; then
    if [ $exit_code -eq $CRITICAL ]; then
        final_exit_code=$CRITICAL
    else
        final_exit_code=$WARNING
    fi
  else
    final_exit_code=$exit_code
  fi
else
  final_exit_code=$exit_code
fi

case $final_exit_code in
  0) status_label="OK" ;;
  1) status_label="WARNING" ;;
  2) status_label="CRITICAL" ;;
  3) status_label="UNKNOWN" ;;
esac

if [ "$sticky" == true ]; then
  new_output=$(echo "$raw_output" | sed -E "s/\b(OK|WARNING|CRITICAL|UNKNOWN)\b/${status_label}/I" | sed '1 { s/^/STICKY /; s/  */ /g; }')

  echo "$new_output"
else
  echo "$raw_output"
fi

exit $final_exit_code
