#!/bin/bash

#Copyright (C) 2026 Sanna Favri Glue AG
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.

#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.

#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <https://www.gnu.org/licenses/>.

#HOW IT WORKS:
#Check_growth checks the growth of a value.
#It uses a statefile for it inside the tmp-directory to get the previous value from and later store the new one in.
#It takes two thresholds: one for critical and one for warning.
#If the growth is higher than any it will return the corresponding text inside the console and the exit-code.
#To work it needs a script to return the new size of the value.
#You can also set options for this script as an argument for the option -o.
#You can either show the output of the script you pass to the -s option or you leave it hidden.
#It works for any other Script/Plugin that returns a numeric value in its performance data
#If the wrapped plugin returns something like LOGFILESIZE | log_size=10000;;;;
#the wrapper will use it for its own output.
#It will use LOGFILESIZE like this: GROWTH LOGFILESIZE STATUS | the rest of the output...
#This will only happen if:
#1. The "Name of the Value" is located on the left side of the pipe.
#2. The name is shorter than 25 characters or everything is written in capital letters.
#If it does not fit one of these cases, it will be treated like additional log-information and will be printed on a new line.
#For example:
#GROWTH STATUS | the rest of the output...
#log-information...

VERSION="2.0"
VERDATE="2026-01-26"
SCRIPT_NAME=$(basename "$0")

#For further explanation about exit-codes for Nagios: https://nagios-plugins.org/doc/guidelines.html#PLUGOUTPUT -> Plugin Return Codes
OK=0
WARNING=1
CRITICAL=2
UNKNOWN=3

#Definition of usage function -> is called if options are incorrect or missing in command
function usage() {
  local msg="$*"
  echo "$SCRIPT_NAME v$VERSION ($VERDATE)"
  echo "$SCRIPT_NAME -s <statefile> | --statefile <statefile> [-t <tmpdirectory> | --tmpdirectory <tmpdirectory> ] -c <critspec> | --critspec <critspec> -w <warnspec> | --warnspec <warnspec> -p <performancedataname> | --performancedataname <performancedataname> [--help] <wrapped_plugin> [wrapped_plugin_options]"
  echo "  -s, --statefile <statefile> File to get previous state from"
  echo "  -t, --tmpdirectory <tmpdirectory> Direcotry to get and save statefile "
  echo "  -c, --critspec <critspec> Critical Growth Threshold"
  echo "  -w, --warnspec <warnspec> Warning Growth Threshold"
  echo "  -p, --performancedataname <performancedataname> Name to get value from performance data"
  if [ -n "$msg" ]; then
    echo "$msg"
  fi
    exit $UNKNOWN
}

#Set default values
tmpdir="/tmp"

#Handling options and arguments
while [[ $# -gt 0 ]]; do
  case $1 in
    -s|--statefile)
      if [ -n "$2" ]; then
        statefile="$2"
      else
        usage "Invalid argument to -s"
      fi
      shift 2
      ;;
    -c|--critspec)
      if [[ "$2" =~ ^[0-9]+$ ]]; then
        critical_threshold="$2"
      else
        usage "Invalid argument to -c"
      fi
      shift 2
      ;;
    -w|--warnspec)
      if [[ "$2" =~ ^[0-9]+$ ]]; then
        warning_threshold="$2"
      else
        usage "Invalid argument to -w"
      fi
      shift 2
      ;;
    -t|--tmpdirectory)
      if [ -d "$2" ] && [ -r "$2" ]; then
        tmpdir="$2"
      else
        usage "$2 is not a writable directory"
      fi
      shift 2
      ;;
    -p|--performancedataname)
      if [ -n "$2" ]; then
        performanceDataName="$2"
      fi
      shift 2
      ;;
    --help)
      usage
      exit
      ;;
    --)
      shift
      break;;
    *)
      break;;
  esac
done

#Check if necessary options are set
[[ -z "$statefile" ]] && usage Must specify -s
[[ -z "$critical_threshold" ]] && usage Must specify -c
[[ -z "$warning_threshold" ]] && usage Must specify -w
[[ -z "$performanceDataName" ]] && usage Must specify -p

#Check if statefile already exists in tmpdir
if [[ ! -f  "${tmpdir}/$(echo "$statefile" | tr '/' '_')_growth_stat" ]]; then
  fileExists=false
else
  fileExists=true
fi

#Safes statefile in variable, gets created first and then safed if it doesn't exist already
statefile="${tmpdir}/$(echo "$statefile" | tr '/' '_')_growth_stat"

#Sets value for oldsize
if [[ -f "$statefile" ]]; then
  oldsize=$(grep '^prvvalue=' "$statefile" | cut -d '=' -f2)
  if [[ ! "$oldsize" =~ ^[0-9]+$ ]]; then
    oldsize=0
  fi
else
  oldsize=0
fi

#Sets value for newsize
script_to_run="$@"
[[ -z "$script_to_run" ]] && usage Must specify a script to run
output=$($script_to_run 2>&1)
perfdata=$(echo "$output" | awk -F'|' '{print $2}')
newsize=$(echo "$perfdata"| grep -oP "(?<=${performanceDataName}=)[^;[:space:]]+")

#Gets the "name of the value" or "additional log"
full_prefix=$(echo "$output" | cut -d'|' -f1)

cleaned=$(echo "$full_prefix" | sed -E 's/\b(OK|WARNING|CRITICAL|UNKNOWN)\b//gI; s/  */ /g; s/^ *//; s/ *$//')

if [[ "$cleaned" =~ ^[A-Z0-9_[:space:]]+$ ]] || [ ${#cleaned} -lt 25 ]; then
    inner_info="$cleaned"
    detailed_log=""
else
    inner_info=""
    detailed_log="$cleaned"
fi

#Checks if newsize is set
if [[ -z "$newsize" ]]; then
    echo "GROWTH ${inner_info} UNKNOWN - Cannot get value from $script_to_run | growth=0;$warning_threshold;$critical_threshold;; $performanceDataName=UNKNOWN;;;;"
    exit $UNKNOWN
elif [ "$fileExists" == false ]; then #checks if Statefile exists
    echo "prvvalue=$newsize" >> "$statefile"
    echo "GROWTH ${inner_info} UNKNOWN - NO VALUE IN $statefile YET | growth=0;$warning_threshold;$critical_threshold;; $performanceDataName=UNKNOWN;;;;"
    exit $UNKNOWN
else
  #Calculates delta
  delta=$((newsize - oldsize))

  #Overwrites old value in statefile and replaces it with newsize
  if grep -q '^prvvalue=' "$statefile"; then
    sed -i 's/^prvvalue=.*/prvvalue='"$newsize"'/' "$statefile"
  else
    echo "prvvalue=$newsize" >> "$statefile"
  fi

  #Separating bing output into variables
  growthPerfData="growth=$delta;$warning_threshold;$critical_threshold;;"
  prfDataData="$performanceDataName=$(echo "$perfdata"| grep -oP "(?<=${performanceDataName}=)[^;[:space:]]+");;;;"
  pDataReturn="$growthPerfData $prfDataData"

  #Checks growth
  if [ "$delta" -gt "$critical_threshold" ]; then
    final_exit_code=$CRITICAL
    status_label="CRITICAL"
  elif [ "$delta" -gt "$warning_threshold" ]; then
    final_exit_code=$WARNING
    status_label="WARNING"
  else
    final_exit_code=$OK
    status_label="OK"
  fi

  if [[ -n "$inner_info" && "$inner_info" != "GROWTH" ]]; then
    echo "GROWTH ${inner_info} ${status_label} - Value has grown by ${delta} | ${pDataReturn}"
  elif [[ -n "$detailed_log" ]]; then
    echo "GROWTH  ${status_label} - Value has grown by ${delta} | ${pDataReturn}"
    echo "${detailed_log}"
  else
    echo "GROWTH ${status_label} - Value has grown by ${delta} | ${pDataReturn}"
  fi

  exit $final_exit_code
fi
