#!/usr/bin/perl
################################################################################
#                                                                              #
#  Copyright (C) 2011 Chad Columbus <ccolumbu@hotmail.com>                     #
#                                                                              #
#   This program is free software; you can redistribute it and/or modify       #
#   it under the terms of the GNU General Public License as published by       #
#   the Free Software Foundation; either version 2 of the License, or          #
#   (at your option) any later version.                                        #
#                                                                              #
#   This program is distributed in the hope that it will be useful,            #
#   but WITHOUT ANY WARRANTY; without even the implied warranty of             #
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              #
#   GNU General Public License for more details.                               #
#                                                                              #
#   You should have received a copy of the GNU General Public License          #
#   along with this program; if not, write to the Free Software                #
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  #
#                                                                              #
################################################################################

use strict;
use DBI;
use Getopt::Std;
$| = 1;

my $VERSION = "Version 1.0";
my $AUTHOR = '(c) 2011 Chad Columbus <ccolumbu@hotmail.com>';

my %opts;
getopts('hH:d:u:p:t:q:', \%opts);

# Exit codes
my $STATE_OK = 0;
my $STATE_WARNING = 1;
my $STATE_CRITICAL = 2;
my $STATE_UNKNOWN = 3;

####################################
####################################
# WARNING WARNING WARNING
# 1. I highly recommend you create a dedicated user for this check that has no privileges except to run the $db_test_query query (i.e. "Show Databases" or similar read only query).
# 2. I highly recommend you don't pass -H or --Host and always force the script to use the default 127.0.0.1 (localhost).
# WARNING WARNING WARNING
####################################
####################################

# Default values:
my $option_count = 0;
my %final;
$final{'db_host'} = '127.0.0.1';
$final{'db_name'} = '';
$final{'db_user'} = '';
$final{'db_passwd'} = '';
$final{'db_type'} = 'mysql';
$final{'db_test_query'} = 'Show Databases';

##################################
# Get command line options

# If called with h or no options show help:
if ($opts{'h'} || scalar(keys %opts) == 0) {
	&print_help();
	exit($STATE_OK);
}

# If a host is provided use it:
if (defined $opts{'H'}) {
	if ($opts{'H'} ne '') {
		$final{'db_host'} = $opts{'H'};
	}
}

# Make sure a database is provided:
if (! defined $opts{'d'}) {
	# SQL user not provided
	print "\nYou must provide a database to connect to. Example: -d mysql\n";
	exit($STATE_UNKNOWN);
} else {
	$final{'db_name'} = $opts{'d'};
}

# Make sure a SQL user is provided:
if (! defined $opts{'u'}) {
	# SQL user not provided
	print "\nYou must provide a SQL user. Example: -u nagios\n";
	exit($STATE_UNKNOWN);
} else {
	$final{'db_user'} = $opts{'u'};
}

# Make sure a SQL passord is provided:
if (! defined $opts{'p'}) {
	# SQL user not provided
	print "\nYou must provide a SQL user's password. Example: -p ######\n";
	exit($STATE_UNKNOWN);
} else {
	$final{'db_passwd'} = $opts{'p'};
}

# If a type is provided use it:
if (defined $opts{'t'}) {
	if ($opts{'t'} ne '') {
		$final{'db_type'} = $opts{'t'};
	}
}

# If a host is provided use it:
if (defined $opts{'q'}) {
	if ($opts{'q'} ne '') {
		$final{'db_test_query'} = $opts{'q'};
	}
}
##################################

##################################
##################################
# Start main code:
our @db_master_connect_string;
$db_master_connect_string[0] = "DBI:$final{'db_type'}:$final{'db_name'};host=$final{'db_host'}";
$db_master_connect_string[1] = "$final{'db_user'}";
$db_master_connect_string[2] = "$final{'db_passwd'}";
$db_master_connect_string[3] = {PrintError => 0, RaiseError => 0, InactiveDestroy => 1, mysql_auto_reconnect => 1};

our $dbh = DBI->connect(@db_master_connect_string);
if (! $dbh) {
	#print STDERR "$0 DBI did not connect, retrying once.\n";
	sleep 1;
	$dbh = DBI->connect(@db_master_connect_string);
}

# Check if we can connect:
if (! $dbh) {
	print "ERROR: Can't connect to DB on Host: $final{'db_host'}, Database: $final{'db_name'}, with User: $final{'db_user'}. ERRSTR: $DBI::errstr \n";
	exit($STATE_CRITICAL);
} else {
	# Set timeout:
	$dbh->do("SET LOCAL net_read_timeout = 30");

	# Check that we can run a query:
	my $sth = $dbh->prepare("$final{'db_test_query'}");
	$sth->execute;
	my $rows = $sth->rows;

	if ($rows <= 0) {
		print "ERROR: Can't run '$final{'db_test_query'}' query. ERRSTR: " . $dbh->errstr . "\n";
		exit($STATE_CRITICAL);
	}
	####################

	$sth->finish;
	$dbh->disconnect();
}

print "check_database_deeply OK - type=$final{'db_type'}, host=$final{'db_host'}, database=$final{'db_name'}, user=$final{'db_user'}\n";
exit($STATE_OK);

####################################
# Start Subs:
####################################
sub print_help() {
        print << 'EOF';
Check a database by connecting to it and running a read only (select) query.
$VERSION
$AUTHOR

Options:
-h
	Print detailed help screen

-H (Optional. Default: 127.0.0.1)
	Hostname or IP

-d (Required)
	Name of the database to connect to for the DBI connection string.

-u (Required)
	SQL username, for the DBI connection string
	#### WARNING ###
	I highly recommend you create a dedicated user for this check.
	A user that has no privileges except to run the test query (i.e. "Show Databases" or similar read only query).
	#### WARNING ###

-p (Required)
	SQL user's password, for the DBI connection string

-t (Optional. Default: mysql)
	Any Perl DBI:<DB_TYPE>. i.e. mysql MariaDB, Pg (PostgreSQL), etc.

-q (Optional. Default: Show Databases)
	### CAUTION ###
	This should ALWAYS be a read only (select) query.
	A read only (select) query to check that the DB will run queries.

###########################
# For security I highly recommend you don't pass -H or --Host and always force the script to use 127.0.0.1 (localhost).
# Example command definition:
define command {
	command_name    check_database_deeply
	command_line    $USER1$/check_database_deeply.pl -d '$ARG1$' -u '$ARG2$' -p '$ARG3$' -t '$ARG4$' -q '$ARG5$'
}

# If you feel it is safe to use this over the network you could use this (but PLEASE don't):
define command {
	command_name    check_database_deeply
	command_line    $USER1$/check_database_deeply.pl -H $HOSTALIAS$ -d '$ARG1$' -u '$ARG2$' -p '$ARG3$' -t '$ARG4$' -q '$ARG5$'
}
# If you need remote checks the right way to do them is to install this script on the remote server and use nrpe to call it using the default 127.0.0.1 host.

###########################
# Example service definition:
define service {
        use			local-service
        host_name		(Hostname)
        service_description	Check Database Deeply
        check_command		check_database_deeply!yourDBname!yourUSERname!yourPASSword!yourDBtype!yourREADONLYquery
}

EOF
}

