# Author: Michael Langevin
# SPDX-License-Identifier: MIT
# Copyright (c) 2025 Michael Langevin
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND...

#!/usr/bin/env python3
"""
Nagios plugin: Check geolocation for a domain or IP using ip-api.com

Exit codes:
  0 - OK
  1 - WARNING
  2 - CRITICAL
  3 - UNKNOWN

Usage:
  check_geo_ip.py -H <hostname_or_ip>

This plugin resolves the given hostname or IP address and fetches its geolocation.
"""

import argparse
import sys
import socket
import requests

# Nagios exit codes
EXIT_OK, EXIT_WARN, EXIT_CRIT, EXIT_UNK = 0, 1, 2, 3

API_URL = "http://ip-api.com/json/{}"

def resolve_host(host):
    try:
        return socket.gethostbyname(host)
    except socket.gaierror as e:
        print(f"CRITICAL - Failed to resolve host '{host}': {e}")
        sys.exit(EXIT_CRIT)

def get_geo_info(ip_address, fields=None):
    params = {}
    if fields:
        params["fields"] = ",".join(fields)

    try:
        response = requests.get(API_URL.format(ip_address), params=params, timeout=5)
        response.raise_for_status()
        return response.json()
    except requests.RequestException as e:
        print(f"UNKNOWN - API request failed: {e}")
        sys.exit(EXIT_UNK)

def main():
    parser = argparse.ArgumentParser(
        description="Nagios plugin to check IP geolocation using ip-api.com"
    )
    parser.add_argument('-H', '--host', required=True, help="Hostname or IP address to geolocate")
    args = parser.parse_args()

    host = args.host
    ip = resolve_host(host)

    fields = [
        "status", "country", "regionName", "city", "org"
    ]

    data = get_geo_info(ip, fields=fields)

    if not data or data.get("status") != "success":
        print("CRITICAL - Failed to retrieve geolocation data.")
        sys.exit(EXIT_CRIT)

    city = data.get("city", "Unknown")
    region = data.get("regionName", "Unknown")
    country = data.get("country", "Unknown")
    org = data.get("org", "Unknown")


    print("OK - Geolocation data retrieved successfully: Results Below! \n"
      f"  IP Address : {ip}\n"
      f"  Location   : {city}, {region}, {country}\n"
      f"  ISP        : {org}")


    sys.exit(EXIT_OK)

if __name__ == "__main__":
    main()
