#!/usr/bin/python3
import requests
import json
import sys
import os
from datetime import datetime
from dotenv import load_dotenv

"""
This plugin uses the Cloudflare API to return tunnel status, as well as other relavent information.
Since this requires an API and account key, you will need to store your keys inside of a .env file. 

Cloudflare API offical documentation: https://developers.cloudflare.com/api/resources/zero_trust/subresources/tunnels/
"""

# Load API credentials from .env file
# Replace with your own .env path or use a standard location.
load_dotenv("/path/to/your/cloudflare_api.env")

# Sends a get request to Cloudflare's API and returns a list of tunnel objects.
def fetch_stats():
    api_key = os.getenv("API_KEY")
    account_id = os.getenv("ACCOUNT_ID")

    if not api_key or not account_id:
        print("UNKNOWN - Missing API_KEY or ACCOUNT_ID in env.")
        sys.exit(3)

    url = f"https://api.cloudflare.com/client/v4/accounts/{account_id}/tunnels"

    headers = {
        'Authorization': f"Bearer {api_key}"
    }

    # Send get request
    try:
        r = requests.get(url, headers=headers)
        r.raise_for_status()
    except requests.exceptions.RequestException as e:
        print(f"UNKNOWN - API request failed: {e}")
        sys.exit(3)


    tunnels = r.json().get("result", [])

    # Filter out tunnels that are deleted.
    tunnels = [t for t in tunnels if t.get("deleted_at") is None]

    return tunnels

# This function takes all of the tunnel objects and prints it in a Nagios XI friendly format.
def print_stats(tunnels):

    if not tunnels:
        print("CRITICAL - No tunnels found")
        sys.exit(2)

    inactive = [t for t in tunnels if t.get("status") != "healthy"]
    active = [t for t in tunnels if t.get("status") == "healthy"]


    if len(active) == len(tunnels):
        print(f"OK - {len(tunnels)} tunnels are active. | active={len(active)} inactive={len(inactive)}")
        exit_code = 0

    elif len(active) > 0:
        print(f"WARNING - {len(inactive)} of {len(tunnels)} tunnels are inactive. | active={len(active)} inactive={len(inactive)}")
        exit_code = 1
        
    else:
        print(f"CRITICAL - All {len(tunnels)} tunnels are inactive. | active={len(active)} inactive={len(inactive)}")
        exit_code = 2


    print("Tunnel Details:")
    print("-" *60)

    for tunnel in tunnels:
        print(f"Name: {tunnel.get('name', 'Unknown')}")
        print(f"Tunnel ID: {tunnel.get('id', 'Unknown')}")
        print(f"Status: {tunnel.get('status', 'Unknown')}")

        if tunnel.get("connections") and len(tunnel["connections"]) > 0:
            conn = tunnel["connections"][0]
            colo = conn.get("colo_name", "Unknown")
            opened = conn.get("opened_at", "Unknown")

            try:
                dt = datetime.strptime(opened, "%Y-%m-%dT%H:%M:%S.%fZ")
                formatted = dt.strftime("%Y-%m-%d %H:%M:%S UTC")
            except (ValueError, TypeError):
                formatted = "Invalid timestamp"

            print(f"Data Center: {colo}")
            print(f"Opened: {formatted}")
        else:
            print("No active connections.")

        print("-" *60)

    return exit_code



def main():
    tunnels = fetch_stats()
    exit_code = print_stats(tunnels)
    sys.exit(exit_code)

if __name__ == "__main__":
    main()
