﻿<#
.SYNOPSIS
    NCPA plugin to check expiration of certificates in a given certificate store.

.DESCRIPTION
    Enumerates certificates in the given store path and checks their expiration.
    Returns Nagios-compatible exit codes.

.PARAMETER StorePath
    Certificate store path, e.g. "Cert:\LocalMachine\My" or "Cert:\CurrentUser\My".

.PARAMETER Warning
    Warning threshold in days (default 30).

.PARAMETER Critical
    Critical threshold in days (default 10).

.EXAMPLE
    .\check_cert_expiry_store.ps1 -StorePath "Cert:\LocalMachine\My" -Warning 30 -Critical 10
#>

param(
    [Parameter(Mandatory=$true)][string]$StorePath,
    [int]$Warning = 30,
    [int]$Critical = 10
)

function Exit-WithCode {
    param([int]$code, [string]$message)
    Write-Output $message
    exit $code
}

try {
    if (-not (Test-Path $StorePath)) {
        Exit-WithCode 3 "UNKNOWN - Store path not found: $StorePath"
    }

    $certs = Get-ChildItem -Path $StorePath
    if (-not $certs) {
        Exit-WithCode 3 "UNKNOWN - No certificates found in $StorePath"
    }

    $results = @()
    $status = 0   # 0=OK, 1=WARNING, 2=CRITICAL, 3=UNKNOWN

    foreach ($cert in $certs) {
        try {
            $daysRemaining = (New-TimeSpan -Start (Get-Date) -End $cert.NotAfter).Days
            $msg = "$($cert.Subject) expires $($cert.NotAfter) ($daysRemaining days left)"

            if ($daysRemaining -le $Critical) {
                $results += "CRITICAL - $msg"
                if ($status -lt 2) { $status = 2 }
            }
            elseif ($daysRemaining -le $Warning) {
                $results += "WARNING - $msg"
                if ($status -lt 1) { $status = 1 }
            }
            else {
                $results += "OK - $msg"
            }
        }
        catch {
            $results += "UNKNOWN - Error reading cert $($cert.Subject): $_"
            $status = 3
        }
    }

    # Return the worst-case status
    Exit-WithCode $status ($results -join " | ")
}
catch {
    Exit-WithCode 3 "UNKNOWN - Script error: $_"
}
